local targetBotLure = false
local targetCount = 0 
local delayValue = 0
local lureMax = 0
local anchorPosition = nil
local lastCall = now
local delayFrom = nil
local dynamicLureDelay = false

-- Debug logging function 
local DEBUG_MODE = false

local function debugLog(level, message)
  if not DEBUG_MODE then return end
  local timestamp = os.date("%H:%M:%S")
  local prefix = string.format("[%s] [TargetBot] [%s]: ", timestamp, level:upper())
  print(prefix .. message)
end

function getWalkableTilesCount(position)
  local count = 0

  for i, tile in pairs(getNearTiles(position)) do
      if tile:isWalkable() or tile:hasCreature() then
          count = count + 1
      end
  end

  return count
end

function rePosition(minTiles)
  minTiles = minTiles or 8
  if now - lastCall < 500 then return end
  local pPos = player:getPosition()
  local tiles = getNearTiles(pPos)
  local playerTilesCount = getWalkableTilesCount(pPos)
  local tilesTable = {}

  if playerTilesCount > minTiles then return end
  for i, tile in ipairs(tiles) do
      tilesTable[tile] = not tile:hasCreature() and tile:isWalkable() and getWalkableTilesCount(tile:getPosition()) or nil
  end

  local best = 0
  local target = nil
  for k,v in pairs(tilesTable) do
      if v > best and v > playerTilesCount then
          best = v
          target = k:getPosition()
      end
  end

  if target then
      lastCall = now
      -- CaveBot removed
  end
end

function findBestEscapePosition(playerPos, creaturePos, minDistance, maxDistance)
  local bestPos = nil
  local bestScore = -999
  
  -- Calculate direction away from creature
  local dx = playerPos.x - creaturePos.x
  local dy = playerPos.y - creaturePos.y
  
  -- Calculate current distance
  local currentDistance = math.max(math.abs(dx), math.abs(dy))
  
  debugLog("info", string.format("Current distance: %d, Target range: %d-%d", currentDistance, minDistance, maxDistance))
  
  -- Function to check if a position is "safe" from the creature
  local function isPositionSafeFromCreature(pos, creaturePos)
    -- Check if creature can easily reach this position in 1-2 moves
    local creaturePath = findPath(creaturePos, pos, 3, {
      ignoreNonPathable = true,
      ignoreCreatures = true
    })
    
    -- If creature can reach in 1 move, definitely unsafe
    if creaturePath and #creaturePath == 1 then
      return false -- Creature can reach this position in 1 move
    end
    
    -- If creature can reach in 2 moves, consider it unsafe for close positions
    if creaturePath and #creaturePath == 2 then
      local distanceToCreature = getDistanceBetween(pos, creaturePos)
      if distanceToCreature <= 2 then
        return false -- Too close, creature can reach easily
      end
    end
    
    -- Also check if position is in creature's immediate attack range
    local distanceToCreature = getDistanceBetween(pos, creaturePos)
    if distanceToCreature <= 1 then
      return false -- Too close to creature
    end
    
    return true -- Position is relatively safe
  end
  
  -- Function to check for obstacles around a position
  local function hasObstacles(pos, radius)
    radius = radius or 1
    local obstacleCount = 0
    
    for x = -radius, radius do
      for y = -radius, radius do
        if x ~= 0 or y ~= 0 then -- Don't check the position itself
          local checkPos = {x = pos.x + x, y = pos.y + y, z = pos.z}
          local tile = g_map.getTile(checkPos)
          if tile and not tile:isWalkable() then
            obstacleCount = obstacleCount + 1
          end
        end
      end
    end
    
    return obstacleCount
  end
  
  -- Function to analyze threat level in each direction around player
  local function analyzeThreatDirections(playerPos, scanRadius)
    local threatMap = {}
    
    -- Initialize threat levels for each direction
    local directions = {
      {name = "North", dx = 0, dy = -1},
      {name = "Northeast", dx = 1, dy = -1},
      {name = "East", dx = 1, dy = 0},
      {name = "Southeast", dx = 1, dy = 1},
      {name = "South", dx = 0, dy = 1},
      {name = "Southwest", dx = -1, dy = 1},
      {name = "West", dx = -1, dy = 0},
      {name = "Northwest", dx = -1, dy = -1}
    }
    
    for _, dir in ipairs(directions) do
      local threatLevel = 0
      local mobCount = 0
      
      -- Scan in this direction for mobs
      for distance = 1, scanRadius do
        local checkPos = {
          x = playerPos.x + (dir.dx * distance),
          y = playerPos.y + (dir.dy * distance),
          z = playerPos.z
        }
        
        local tile = g_map.getTile(checkPos)
        if tile then
          local creature = tile:getTopCreature()
          if creature and creature:isMonster() then
            mobCount = mobCount + 1
            -- Closer mobs are more threatening
            threatLevel = threatLevel + (scanRadius - distance + 1) * 2
          end
        end
      end
      
      threatMap[dir.name] = {
        threatLevel = threatLevel,
        mobCount = mobCount,
        direction = dir
      }
      
      debugLog("info", string.format("Direction %s: threat=%d, mobs=%d", 
        dir.name, threatLevel, mobCount))
    end
    
    return threatMap
  end
  
  -- Function to find safest direction to escape
  local function findSafestDirection(playerPos, creaturePos, threatMap)
    local safestDirection = nil
    local lowestThreat = 999
    
    -- Calculate direction away from target creature
    local dx = playerPos.x - creaturePos.x
    local dy = playerPos.y - creaturePos.y
    
    -- Normalize direction
    local targetDirX = dx > 0 and 1 or (dx < 0 and -1 or 0)
    local targetDirY = dy > 0 and 1 or (dy < 0 and -1 or 0)
    
    debugLog("info", string.format("Target creature direction: %d,%d", targetDirX, targetDirY))
    
    -- Find direction with lowest threat that also moves away from target
    for dirName, threatData in pairs(threatMap) do
      local dir = threatData.direction
      
      -- Bonus for moving away from target creature
      local awayFromTargetBonus = 0
      if (targetDirX == 0 or dir.dx == targetDirX) and (targetDirY == 0 or dir.dy == targetDirY) then
        awayFromTargetBonus = 50 -- Bonus for moving away from target
      elseif (targetDirX ~= 0 and dir.dx == -targetDirX) or (targetDirY ~= 0 and dir.dy == -targetDirY) then
        awayFromTargetBonus = -30 -- Penalty for moving toward target
      end
      
      local adjustedThreat = threatData.threatLevel - awayFromTargetBonus
      
      debugLog("info", string.format("Direction %s: base threat=%d, adjusted=%d (away bonus=%d)", 
        dirName, threatData.threatLevel, adjustedThreat, awayFromTargetBonus))
      
      if adjustedThreat < lowestThreat then
        lowestThreat = adjustedThreat
        safestDirection = dir
      end
    end
    
    debugLog("info", string.format("Safest direction: %d,%d (threat level: %d)", 
      safestDirection.dx, safestDirection.dy, lowestThreat))
    
    return safestDirection
  end
  
  -- Analyze threat directions around player
  local scanRadius = minDistance + maxDistance -- Scan across entire distance range
  debugLog("info", string.format("Analyzing threats within radius %d (keepDistance=%d + maxDistance=%d)", scanRadius, minDistance, maxDistance))
  
  local threatMap = analyzeThreatDirections(playerPos, scanRadius)
  local safestDirection = findSafestDirection(playerPos, creaturePos, threatMap)
  
  -- Scan positions in the safest direction first, then expand outward
  debugLog("info", string.format("Prioritizing escape in direction %d,%d", safestDirection.dx, safestDirection.dy))
  
  -- Scan in expanding rings, but prioritize the safest direction
  for radius = 1, scanRadius do
    local foundGoodPosition = false
    
    -- First, check positions in the safest direction
    local priorityPositions = {}
    
    -- Add positions in safest direction
    for distance = radius, radius do
      local testPos = {
        x = playerPos.x + (safestDirection.dx * distance),
        y = playerPos.y + (safestDirection.dy * distance),
        z = playerPos.z
      }
      table.insert(priorityPositions, testPos)
    end
    
    -- Add other positions on the ring
    for x = -radius, radius do
      for y = -radius, radius do
        -- Only check positions on the current ring (perimeter)
        if math.abs(x) == radius or math.abs(y) == radius then
          local testPos = {
            x = playerPos.x + x,
            y = playerPos.y + y,
            z = playerPos.z
          }
          
          -- Skip if already in priority list
          local alreadyAdded = false
          for _, priorityPos in ipairs(priorityPositions) do
            if priorityPos.x == testPos.x and priorityPos.y == testPos.y then
              alreadyAdded = true
              break
            end
          end
          
          if not alreadyAdded then
            table.insert(priorityPositions, testPos)
          end
        end
      end
    end
    
    -- Evaluate all positions
    for _, testPos in ipairs(priorityPositions) do
      local distanceToCreature = getDistanceBetween(testPos, creaturePos)
      
      -- Only consider positions within our target distance range
      if distanceToCreature >= minDistance and distanceToCreature <= maxDistance then
        local tile = g_map.getTile(testPos)
        if tile and tile:isWalkable() and not tile:hasCreature() then
          local score = 0
          
          -- Check if position is safe from creature (simplified for speed)
          local isSafe = distanceToCreature > 2 -- Simple safety check
          
          -- Heavy bonus for safe positions
          if isSafe then
            score = score + 100 -- Major bonus for avoiding creature
          else
            score = score - 50 -- Penalty for unsafe positions
          end
          
          -- Bonus for being in the safest direction
          local dirX = testPos.x - playerPos.x
          local dirY = testPos.y - playerPos.y
          if dirX ~= 0 and dirY ~= 0 then
            dirX = dirX > 0 and 1 or -1
            dirY = dirY > 0 and 1 or -1
            if dirX == safestDirection.dx and dirY == safestDirection.dy then
              score = score + 50 -- Bonus for safest direction
            end
          elseif dirX ~= 0 then
            dirX = dirX > 0 and 1 or -1
            if dirX == safestDirection.dx and safestDirection.dy == 0 then
              score = score + 50 -- Bonus for safest direction
            end
          elseif dirY ~= 0 then
            dirY = dirY > 0 and 1 or -1
            if dirY == safestDirection.dy and safestDirection.dx == 0 then
              score = score + 50 -- Bonus for safest direction
            end
          end
          
          -- Simplified obstacle check (only check immediate neighbors)
          local obstacleCount = 0
          for ox = -1, 1 do
            for oy = -1, 1 do
              if ox ~= 0 or oy ~= 0 then
                local obsPos = {x = testPos.x + ox, y = testPos.y + oy, z = testPos.z}
                local obsTile = g_map.getTile(obsPos)
                if obsTile and not obsTile:isWalkable() then
                  obstacleCount = obstacleCount + 1
                end
              end
            end
          end
          
          if obstacleCount == 0 then
            score = score + 30 -- Bonus for clear area
          elseif obstacleCount <= 2 then
            score = score + 10 -- Small bonus for mostly clear area
          else
            score = score - 20 -- Penalty for cluttered area
          end
          
          -- Bonus for optimal distance (prefer middle of range)
          local optimalDistance = (minDistance + maxDistance) / 2
          local distanceFromOptimal = math.abs(distanceToCreature - optimalDistance)
          score = score + math.max(0, 20 - distanceFromOptimal * 5)
          
          -- Quick path check (limited for performance)
          local path = findPath(playerPos, testPos, 5, {
            ignoreNonPathable = true,
            ignoreCreatures = true,
            ignoreCost = true,
            ignoreLastCreature = true
          })
          
          if path and #path <= 5 then -- Shorter path limit for performance
            score = score + 15 -- Bonus for clear path
            debugLog("info", string.format("Position %d,%d: distance=%d, score=%d, safe=%s, obstacles=%d, path=%d", 
              testPos.x, testPos.y, distanceToCreature, score, tostring(isSafe), obstacleCount, #path))
            
            if score > bestScore then
              bestScore = score
              bestPos = testPos
              foundGoodPosition = true
            end
          end
        end
      end
    end
    
    -- If we found a good position in this ring, we can stop scanning
    if foundGoodPosition and bestScore > 50 then
      debugLog("info", string.format("Found good position in ring %d, stopping scan", radius))
      break
    end
  end
  
  -- If we found a good position in the scan, use it
  if bestPos and bestScore > 0 then
    debugLog("info", string.format("Found optimal position: %d,%d,%d (score: %d)", 
      bestPos.x, bestPos.y, bestPos.z, bestScore))
    return bestPos
  end
  
  -- Fallback: if no position found in target range, find the best available position
  debugLog("info", "No position found in target range, looking for best available position")
  bestScore = -999
  bestPos = nil
  
  -- Quick fallback scan - only check adjacent positions for speed
  local directions = {
    {-1, -1}, {-1, 0}, {-1, 1},
    {0, -1},           {0, 1},
    {1, -1},  {1, 0},  {1, 1}
  }
  
  for _, dir in ipairs(directions) do
    local testPos = {
      x = playerPos.x + dir[1],
      y = playerPos.y + dir[2],
      z = playerPos.z
    }
    
    local distanceToCreature = getDistanceBetween(testPos, creaturePos)
    local tile = g_map.getTile(testPos)
    
    if tile and tile:isWalkable() and not tile:hasCreature() and distanceToCreature > currentDistance then
      local score = 0
      
      -- Simple safety check
      local isSafe = distanceToCreature > 2
      
      -- Bonus for safe positions
      if isSafe then
        score = score + 50
      else
        score = score - 20
      end
      
      -- Bonus for increasing distance
      score = score + (distanceToCreature - currentDistance) * 10
      
      -- Quick path check
      local path = findPath(playerPos, testPos, 3, {
        ignoreNonPathable = true,
        ignoreCreatures = true,
        ignoreCost = true,
        ignoreLastCreature = true
      })
      
      if path and #path <= 3 then
        score = score + 10
        debugLog("info", string.format("Fallback pos %d,%d: distance=%d, score=%d, safe=%s", 
          testPos.x, testPos.y, distanceToCreature, score, tostring(isSafe)))
        
        if score > bestScore then
          bestScore = score
          bestPos = testPos
        end
      end
    end
  end
  
  debugLog("info", string.format("Best escape position: %s (score: %d)", 
    bestPos and string.format("%d,%d,%d", bestPos.x, bestPos.y, bestPos.z) or "none", bestScore))
  
  return bestPos
end

TargetBot.Creature.attack = function(params, targets, isLooting) -- params {config, creature, danger, priority}
  if player:isWalking() then
    lastWalk = now
  end

  local config = params.config
  local creature = params.creature
  
  if g_game.getAttackingCreature() ~= creature then
    g_game.attack(creature)
  end

  if not isLooting then -- walk only when not looting
    TargetBot.Creature.walk(creature, config, targets)
  end

  -- attacks
  local mana = player:getMana()
  if config.useGroupAttack and config.groupAttackSpell:len() > 1 and mana > config.minManaGroup then
    local creatures = g_map.getSpectatorsInRange(player:getPosition(), false, config.groupAttackRadius, config.groupAttackRadius)
    local playersAround = false
    local monsters = 0
    for _, creature in ipairs(creatures) do
      if not creature:isLocalPlayer() and creature:isPlayer() and (not config.groupAttackIgnoreParty or creature:getShield() <= 2) then
        playersAround = true
      elseif creature:isMonster() then
        monsters = monsters + 1
      end
    end
    if monsters >= config.groupAttackTargets and (not playersAround or config.groupAttackIgnorePlayers) then
      if TargetBot.sayAttackSpell(config.groupAttackSpell, config.groupAttackDelay) then
        return
      end
    end
  end

  if config.useGroupAttackRune and config.groupAttackRune > 100 then
    local creatures = g_map.getSpectatorsInRange(creature:getPosition(), false, config.groupRuneAttackRadius, config.groupRuneAttackRadius)
    local playersAround = false
    local monsters = 0
    for _, creature in ipairs(creatures) do
      if not creature:isLocalPlayer() and creature:isPlayer() and (not config.groupAttackIgnoreParty or creature:getShield() <= 2) then
        playersAround = true
      elseif creature:isMonster() then
        monsters = monsters + 1
      end
    end
    if monsters >= config.groupRuneAttackTargets and (not playersAround or config.groupAttackIgnorePlayers) then
      if TargetBot.useAttackItem(config.groupAttackRune, 0, creature, config.groupRuneAttackDelay) then
        return
      end
    end
  end
  if config.useSpellAttack and config.attackSpell:len() > 1 and mana > config.minMana then
    if TargetBot.sayAttackSpell(config.attackSpell, config.attackSpellDelay) then
      return
    end
  end
  if config.useRuneAttack and config.attackRune > 100 then
    if TargetBot.useAttackItem(config.attackRune, 0, creature, config.attackRuneDelay) then
      return
    end
  end
end

TargetBot.Creature.walk = function(creature, config, targets)
  local cpos = creature:getPosition()
  local pos = player:getPosition()
  
  local isTrapped = true
  local pos = player:getPosition()
  local dirs = {{-1,1}, {0,1}, {1,1}, {-1, 0}, {1, 0}, {-1, -1}, {0, -1}, {1, -1}}
  for i=1,#dirs do
    local tile = g_map.getTile({x=pos.x-dirs[i][1],y=pos.y-dirs[i][2],z=pos.z})
    if tile and tile:isWalkable(false) then
      isTrapped = false
    end
  end

  -- data for external dynamic lure
  if config.lureMin and config.lureMax and config.dynamicLure then
    if config.lureMin >= targets then
      targetBotLure = true
    elseif targets >= config.lureMax then
      targetBotLure = false
    end
  end
  targetCount = targets
  delayValue = config.lureDelay

  if config.lureMax then
    lureMax = config.lureMax
  end

  dynamicLureDelay = config.dynamicLureDelay
  delayFrom = config.delayFrom

  -- luring
  if config.closeLure and config.closeLureAmount <= getMonsters(1) then
    return
  end
  if TargetBot.canLure() and (config.lure or config.dynamicLure) and not (creature:getHealthPercent() < (storage.extras.killUnder or 30)) and not isTrapped then
    if targetBotLure then
      anchorPosition = nil
      return
    else
      if targets < config.lureCount then
        local path = findPath(pos, cpos, 5, {ignoreNonPathable=true, precision=2})
        if path then
          return TargetBot.walkTo(cpos, 10, {marginMin=5, marginMax=6, ignoreNonPathable=true})
        end
      end
    end
  end

  local currentDistance = findPath(pos, cpos, 10, {ignoreCreatures=true, ignoreNonPathable=true, ignoreCost=true})
  if (not config.chase or #currentDistance == 1) and not config.avoidAttacks and not config.keepDistance and config.rePosition and (creature:getHealthPercent() >= storage.extras.killUnder) then
    return rePosition(config.rePositionAmount or 6)
  end
  if config.chase and not config.keepDistance then
    if #currentDistance > 1 then
      return TargetBot.walkTo(cpos, 10, {ignoreNonPathable=true, precision=1})
    end
  elseif config.keepDistance then
    if not anchorPosition or distanceFromPlayer(anchorPosition) > config.anchorRange then
      anchorPosition = pos
    end
    
    -- Debug logging for keep distance
    local distanceToTarget = #currentDistance
    local minDistance = config.keepDistanceRange
    local maxDistance = config.keepDistanceRange + 1
    
    debugLog("info", string.format("Keep Distance: Current=%d, Min=%d, Max=%d, Creature=%s", 
      distanceToTarget, minDistance, maxDistance, creature:getName()))
    
    -- Move away if too close (more aggressive threshold)
    if distanceToTarget <= minDistance then
      debugLog("info", string.format("Too close! Moving away from %s (distance: %d <= %d)", 
        creature:getName(), distanceToTarget, minDistance))
      
      -- Find best position to move away from creature
      local bestEscapePos = findBestEscapePosition(pos, cpos, minDistance, maxDistance)
      if bestEscapePos then
        debugLog("info", string.format("Found escape position: %d,%d,%d", 
          bestEscapePos.x, bestEscapePos.y, bestEscapePos.z))
        
        -- Force immediate movement to escape position
        local path = findPath(pos, bestEscapePos, 10, {
          ignoreNonPathable = true,
          ignoreCreatures = true,
          ignoreCost = true,
          ignoreLastCreature = true
        })
        
        if path and #path > 0 then
          debugLog("info", string.format("Moving to escape position via path of %d steps", #path))
          
          -- Try direct walking first
          if #path == 1 then
            debugLog("info", "Direct walk to escape position")
            walk(path[1])
            return
          else
            debugLog("info", "Using TargetBot.walkTo for multi-step movement")
            if config.anchor and anchorPosition and getDistanceBetween(bestEscapePos, anchorPosition) <= config.anchorRange*2 then
              TargetBot.walkTo(bestEscapePos, 10, {ignoreNonPathable=true, precision=1})
            else
              TargetBot.walkTo(bestEscapePos, 10, {ignoreNonPathable=true, precision=1})
            end
            return
          end
        else
          debugLog("warning", "No path found to escape position!")
        end
      else
        debugLog("warning", "No escape position found!")
      end
    -- Move closer if too far (less aggressive)
    elseif distanceToTarget > maxDistance then
      debugLog("info", string.format("Too far! Moving closer to %s (distance: %d > %d)", 
        creature:getName(), distanceToTarget, maxDistance))
      
      if config.anchor and anchorPosition and getDistanceBetween(pos, anchorPosition) <= config.anchorRange*2 then
        return TargetBot.walkTo(cpos, 10, {ignoreNonPathable=true, marginMin=minDistance, marginMax=maxDistance, maxDistanceFrom={anchorPosition, config.anchorRange}})
      else
        return TargetBot.walkTo(cpos, 10, {ignoreNonPathable=true, marginMin=minDistance, marginMax=maxDistance})
      end
    else
      debugLog("info", string.format("Distance OK: %d (range: %d-%d)", 
        distanceToTarget, minDistance, maxDistance))
    end
  end

  --target only movement
  if config.avoidAttacks then
    local diffx = cpos.x - pos.x
    local diffy = cpos.y - pos.y
    local candidates = {}
    if math.abs(diffx) == 1 and diffy == 0 then
      candidates = {{x=pos.x, y=pos.y-1, z=pos.z}, {x=pos.x, y=pos.y+1, z=pos.z}}
    elseif diffx == 0 and math.abs(diffy) == 1 then
      candidates = {{x=pos.x-1, y=pos.y, z=pos.z}, {x=pos.x+1, y=pos.y, z=pos.z}}
    end
    for _, candidate in ipairs(candidates) do
      local tile = g_map.getTile(candidate)
      if tile and tile:isWalkable() then
        return TargetBot.walkTo(candidate, 2, {ignoreNonPathable=true})
      end
    end
  elseif config.faceMonster then
    local diffx = cpos.x - pos.x
    local diffy = cpos.y - pos.y
    local candidates = {}
    if diffx == 1 and diffy == 1 then
      candidates = {{x=pos.x+1, y=pos.y, z=pos.z}, {x=pos.x, y=pos.y-1, z=pos.z}}
    elseif diffx == -1 and diffy == 1 then
      candidates = {{x=pos.x-1, y=pos.y, z=pos.z}, {x=pos.x, y=pos.y-1, z=pos.z}}
    elseif diffx == -1 and diffy == -1 then
      candidates = {{x=pos.x, y=pos.y-1, z=pos.z}, {x=pos.x-1, y=pos.y, z=pos.z}} 
    elseif diffx == 1 and diffy == -1 then
      candidates = {{x=pos.x, y=pos.y-1, z=pos.z}, {x=pos.x+1, y=pos.y, z=pos.z}}       
    else
      local dir = player:getDirection()
      if diffx == 1 and dir ~= 1 then turn(1)
      elseif diffx == -1 and dir ~= 3 then turn(3)
      elseif diffy == 1 and dir ~= 2 then turn(2)
      elseif diffy == -1 and dir ~= 0 then turn(0)
      end
    end
    for _, candidate in ipairs(candidates) do
      local tile = g_map.getTile(candidate)
      if tile and tile:isWalkable() then
        return TargetBot.walkTo(candidate, 2, {ignoreNonPathable=true})
      end
    end
  end
end

-- CaveBot position change handler removed